library(shiny)
library(httr)
library(jsonlite)
library(plotly)
library(DT)

# FUNCIONES AUXILIARES

# Está función obtiene los datos del ISS para actualizarlos
actualizar_posicion_iss <- function() {
  # url para obtener los datos
  url <- "https://api.wheretheiss.at/v1/satellites/25544"
  # Hace la peticion para obtener los datos El timeout es un tiempo de espera si tarda mas de 5 s se corta y manda error esto es para evitar que se quede cargando
  r <- GET(url, timeout(5))
  # Comprueba el estado y para en caso de error
  stop_for_status(r)
  # Guardamos la respuesta y la decodificamos
  x <- fromJSON(content(r, as = "text", encoding = "UTF-8"))
  
  # Creamos un dataframe de los datos
  data.frame(
    timestamp = as.numeric(x$timestamp),
    time_utc  = as.POSIXct(as.numeric(x$timestamp), origin = "1970-01-01", tz = "UTC"),
    lat       = as.numeric(x$latitude),
    lon       = as.numeric(x$longitude),
    altitude_km = as.numeric(x$altitude),
    velocity_kmh = as.numeric(x$velocity)
  )
}

# INTERFAZ
ui <- fluidPage(
  titlePanel("ISS Tracker"),
  
  sidebarLayout(
    sidebarPanel(
      # Este slider es para establecer cada cuantos segundos el usuario quiere que se actualizen los datos
      sliderInput("refresh", "Refresco (segundos)", min = 1, max = 10, value = 3, step = 1),
      
      # Botones para pausar o continuar
      fluidRow(
        column(6, actionButton("start", "Start", class = "btn-success", style = "width:100%")),
        column(6, actionButton("stop",  "Stop",  class = "btn-danger",  style = "width:100%"))
      ),
      
      # Aqui meto un separador horizontal para dividir la interfaz
      tags$hr(),
      
      # Botones para limpiar y descargar el csv
      actionButton("clear", "Limpiar trayectoria", style = "width:100%"),
      downloadButton("descargar_csv", "Descargar CSV", style = "width:100%; margin-top:8px;"),
      
      # Otro separador
      tags$hr(),
      
      # Aqui ponemos el log para indicar si hay algún error o algo y así que el usuario lo sepa
      h4("Log"),
      verbatimTextOutput("status", placeholder = TRUE),
      
      # Muestra los últimos datos obtenidos
      h4("Datos Actuales"),
      tags$ul(
        tags$li(strong("Velocidad: "), textOutput("velocidad_actual", inline = TRUE)),
        tags$li(strong("Altitud: "), textOutput("altitud_actual", inline = TRUE)),
        tags$li(strong("Puntos guardados: "), textOutput("num_puntos", inline = TRUE))
      )
    ),
    
    # El panel principal con el mapa, velocidad y la tabla de datos
    mainPanel(
      tabsetPanel(
        tabPanel("Mapa", plotlyOutput("map", height = "520px")),
        tabPanel("Velocidad", plotlyOutput("speed", height = "420px")),
        tabPanel("Tabla", DTOutput("table"))
      )
    )
  )
)

# SERVIDOR
server <- function(input, output, session) {
  
  # Es el estado interno aqui muestra el estado si esta parado o no 
  rv <- reactiveValues(
    running = FALSE,
    last_msg = "Parado",
    # El siguiente dataframe es cada punto que vamos guardando
    track = data.frame(
      timestamp=numeric(), time_utc=as.POSIXct(character(), tz="UTC"),
      lat=numeric(), lon=numeric(),
      altitude_km=numeric(), velocity_kmh=numeric(),
      stringsAsFactors = FALSE
    )
  )
  
  # Este es el bloque que inicia la aplicación es decir cuando el usuario le da a start
  observeEvent(input$start, {
    # Primero modificamos el log para indicar que se ha iniciado
    rv$running <- TRUE
    rv$last_msg <- "Tracking ON"
    # Intentamos pedir un punto a la api. Aqui hemos usado un tryCatch por si se rompe debido a errores de conexión sobretodo
    pt <- tryCatch(actualizar_posicion_iss(), error = function(e) NULL)
    # Si no ha fallado es decri no es nulo seguimos
    if (!is.null(pt)) {
      rv$track <- rbind(rv$track, pt)
      rv$last_msg <- paste0("OK (primer punto: ", format(pt$time_utc, "%H:%M:%S UTC"), ")")
    } else {
      rv$last_msg <- "No pude contactar con la API (revisa conexión / firewall)"
    }
  })
  
  # Actualizamos el log cuando dejamos de seguir
  observeEvent(input$stop, {
    rv$running <- FALSE
    rv$last_msg <- "Tracking OFF"
  })
  
  # Actualizamos el log cuando limpiamos la trayectoria
  observeEvent(input$clear, {
    rv$track <- rv$track[0, ]
    rv$last_msg <- "Trayectoria limpiada"
  })
  
  observe({
    req(rv$running)
    invalidateLater(input$refresh * 1000, session)
    
    pt <- tryCatch(actualizar_posicion_iss(), error = function(e) NULL)
    if (is.null(pt)) {
      rv$last_msg <- "Error llamando a la API (reintentando...)"
      return()
    }
    
    rv$track <- rbind(rv$track, pt)
    
    rv$last_msg <- paste0("OK (último update: ", format(pt$time_utc, "%H:%M:%S UTC"), ")")
  })
  
  # Genera el texto del log
  output$status <- renderText({
    paste(rv$last_msg, paste0("Running: ", rv$running), sep = "\n")
  })
  
  # El número de puntos que llevamos guardados
  output$num_puntos <- renderText(nrow(rv$track))
  
  # Muestra la altitud si no hay puntos muestra un -
  output$altitud_actual <- renderText({
    if (nrow(rv$track) < 1) return("—")
    a <- rv$track$altitude_km[nrow(rv$track)]
    if (is.na(a)) "—" else paste0(round(a, 1), " km")
  })
  
  # Muestra la velocidad si no hay puntos muestra un -
  output$velocidad_actual <- renderText({
    if (nrow(rv$track) < 1) return("—")
    v <- rv$track$velocity_kmh[nrow(rv$track)]
    if (is.na(v)) "—" else paste0(round(v, 0), " km/h")
  })
  
  # Mapa (en el caso de que no haya datos muestra el texto del comienzo"
  output$map <- renderPlotly({
    if (nrow(rv$track) < 1) {
      return(plotly_empty() %>% layout(title = "Pulsa Start para cargar la posición del ISS"))
    }
    
    df <- rv$track
    last <- df[nrow(df), ]
    
    plot_ly() %>%
      add_trace(
        data = df,
        type = "scattergeo",
        mode = "lines+markers",
        lon = ~lon, lat = ~lat,
        text = ~paste0(
          "UTC: ", time_utc,
          "<br>Lat: ", round(lat, 3),
          "<br>Lon: ", round(lon, 3),
          "<br>Alt: ", round(altitude_km, 1), " km",
          "<br>Vel: ", round(velocity_kmh, 0), " km/h"
        ),
        hoverinfo = "text",
        name = "Trayectoria"
      ) %>%
      add_trace(
        data = last,
        type = "scattergeo",
        mode = "markers",
        lon = ~lon, lat = ~lat,
        marker = list(size = 10),
        name = "Última posición"
      ) %>%
      layout(
        geo = list(
          projection = list(type = "natural earth"),
          showland = TRUE,
          showcountries = TRUE
        ),
        margin = list(l=10, r=10, t=30, b=10),
        legend = list(orientation = "h")
      )
  })
  
  # Grafico de la velocidad
  output$speed <- renderPlotly({
    df <- rv$track
    if (nrow(df) < 1) {
      return(plotly_empty(type = "scatter", mode = "lines") %>% layout(title = "Pulsa Start para ver la velocidad"))
    }
    
    plot_ly(
      data = df,
      x = ~time_utc,
      y = ~velocity_kmh,
      type = "scatter",
      mode = "lines+markers",
      hovertemplate = "UTC: %{x}<br>Velocidad: %{y:.0f} km/h<extra></extra>"
    ) %>%
      layout(
        title = "Velocidad (km/h)",
        xaxis = list(title = "Tiempo (UTC)"),
        yaxis = list(title = "km/h"),
        margin = list(l=60, r=10, t=50, b=60)
      )
  })
  
  # Muestra la tabla
  output$table <- renderDT({
    datatable(
      rv$track,
      options = list(pageLength = 10, scrollX = TRUE),
      rownames = FALSE
    )
  })
  
  # Descargar el CSV
  output$descargar_csv <- downloadHandler(
    filename = function() paste0("trayectoria_iss_", format(Sys.time(), "%Y%m%d_%H%M%S"), ".csv"),
    content = function(file) write.csv(rv$track, file, row.names = FALSE)
  )
}

shinyApp(ui, server)
